// State management
let isActive = false;
let currentHighlight = null;

// Listen for messages from background script
browser.runtime.onMessage.addListener((message) => {
  if (message.action === "activate") {
    toggleTableSelection();
  }
});

function toggleTableSelection() {
  if (isActive) {
    deactivate();
  } else {
    activate();
  }
}

function activate() {
  isActive = true;
  document.addEventListener('mouseover', handleMouseOver);
  document.addEventListener('mouseout', handleMouseOut);
  document.addEventListener('click', handleClick);
  showStatus('Hover over a table and click to convert it to CSV');
}

function deactivate() {
  isActive = false;
  document.removeEventListener('mouseover', handleMouseOver);
  document.removeEventListener('mouseout', handleMouseOut);
  document.removeEventListener('click', handleClick);
  if (currentHighlight) {
    currentHighlight.classList.remove('table-to-csv-highlight');
    currentHighlight = null;
  }
}

/**
 * ESPN uses a split layout: a flex wrapper with a fixed-left table (e.g. RK, Name)
 * and a scrollable table (e.g. POS, GP, MIN, PTS...). Returns the wrapper if this
 * table is part of that pattern, so we can merge both tables into one CSV.
 */
function getESPNTableWrapper(table) {
  const flex = table.closest('.flex');
  if (!flex) return null;
  const tables = flex.querySelectorAll('table');
  if (tables.length < 2) return null;
  return flex;
}

/**
 * *-Reference sites (e.g. Basketball-Reference) use leaderboard_grid with one div per metric
 * (id="leaders_g", "leaders_mp", ...). Each is a "table" with h4 = metric name, rows = rank, who, value.
 */
function getReferenceLeaderBlock(element) {
  const grid = element.closest('.leaderboard_grid');
  if (!grid) return null;
  const block = element.closest('div[id^="leaders_"]');
  if (!block || !grid.contains(block)) return null;
  return block;
}

/**
 * Returns the element to highlight and use for CSV: Reference leader block, ESPN wrapper, or table.
 */
function getTableTarget(element) {
  const refBlock = getReferenceLeaderBlock(element);
  if (refBlock) return refBlock;
  const table = element.closest('table');
  if (!table) return null;
  const wrapper = getESPNTableWrapper(table);
  return wrapper || table;
}

function handleMouseOver(e) {
  const target = getTableTarget(e.target);
  if (target && target !== currentHighlight) {
    if (currentHighlight) {
      currentHighlight.classList.remove('table-to-csv-highlight');
    }
    target.classList.add('table-to-csv-highlight');
    currentHighlight = target;
  }
}

function handleMouseOut(e) {
  const target = getTableTarget(e.target);
  const relatedTarget = getTableTarget(e.relatedTarget);
  if (target && target !== relatedTarget) {
    target.classList.remove('table-to-csv-highlight');
    if (currentHighlight === target) {
      currentHighlight = null;
    }
  }
}

function handleClick(e) {
  const target = getTableTarget(e.target);
  if (target) {
    e.preventDefault();
    e.stopPropagation();
    deactivate();
    let csv;
    if (target.id && target.id.startsWith('leaders_') && target.closest('.leaderboard_grid')) {
      csv = referenceLeaderToCSV(target);
    } else if (target.classList.contains('flex')) {
      csv = espnTablesToCSV(target);
    } else {
      csv = tableToCSV(target);
    }
    showCSVDialog(csv);
  }
}

/**
 * Get clean text from a cell. For ESPN athlete cells, prefer the link text (player name).
 */
function getCellText(cell) {
  const link = cell.querySelector('a.AnchorLink');
  if (link) {
    let text = (link.textContent || '').trim();
    const teamAbbrev = cell.querySelector('.athleteCell__teamAbbrev');
    if (teamAbbrev) {
      const team = (teamAbbrev.textContent || '').trim();
      if (team) text = text ? `${text} (${team})` : team;
    }
    if (text) return text;
  }
  let text = cell.textContent || '';
  return text.trim();
}

/**
 * Extract one row of CSV cells from a <tr>, using getCellText for each td/th.
 */
function rowToCells(row) {
  const cells = [];
  const cellElements = row.querySelectorAll('td, th');
  cellElements.forEach(cell => {
    let text = getCellText(cell);
    text = escapeCSVValue(text);
    const colspan = parseInt(cell.getAttribute('colspan'), 10) || 1;
    for (let i = 0; i < colspan; i++) {
      cells.push(text);
    }
  });
  return cells;
}

function tableToCSV(table) {
  const rows = [];
  const allRows = table.querySelectorAll('tr');
  allRows.forEach(row => {
    const cells = rowToCells(row);
    if (cells.length > 0) {
      rows.push(cells.join(','));
    }
  });
  return rows.join('\n');
}

/**
 * ESPN layout: .flex contains table (fixed left: RK, Name) + .Table__ScrollerWrapper with table (POS, GP, MIN, ...).
 * Each table has thead + tbody. Merge: one header row from both theads, then body rows aligned by index.
 */
function espnTablesToCSV(flexWrapper) {
  const tables = Array.from(flexWrapper.querySelectorAll('table'));
  if (tables.length === 0) return '';
  if (tables.length === 1) return tableToCSV(tables[0]);

  const mergedRows = [];

  // Header row: merge thead rows from each table
  const headerCells = [];
  for (const table of tables) {
    const headerRow = table.querySelector('thead tr');
    if (headerRow) {
      headerCells.push(...rowToCells(headerRow));
    }
  }
  if (headerCells.length > 0) {
    mergedRows.push(headerCells.join(','));
  }

  // Body rows: merge tbody rows by index (data-idx or row order)
  const bodyRowCount = Math.max(...tables.map(t => t.querySelectorAll('tbody tr').length));
  for (let i = 0; i < bodyRowCount; i++) {
    const rowCells = [];
    for (const table of tables) {
      const tbodyRows = table.querySelectorAll('tbody tr');
      const row = tbodyRows[i];
      if (row) {
        rowCells.push(...rowToCells(row));
      }
    }
    if (rowCells.length > 0) {
      mergedRows.push(rowCells.join(','));
    }
  }

  return mergedRows.join('\n');
}

/**
 * *-Reference leader block: div[id^="leaders_"] with h4 (metric name), then rows of .rank, .who, .value.
 * CSV: header "Rank", "Player", "<MetricName>"; one data row per leader row.
 */
function referenceLeaderToCSV(block) {
  const h4 = block.querySelector('h4');
  const metricName = (h4 && h4.textContent || 'Value').trim();
  const header = ['Rank', 'Player', escapeCSVValue(metricName)].join(',');

  const wrapper = block.querySelector('div');
  if (!wrapper) return header;
  const rowDivs = Array.from(wrapper.children).filter(
    el => el.querySelector('.rank') && el.querySelector('.who') && el.querySelector('.value')
  );

  const rows = rowDivs.map(rowEl => {
    const rank = (rowEl.querySelector('.rank').textContent || '').trim();
    const whoEl = rowEl.querySelector('.who');
    const link = whoEl && whoEl.querySelector('a');
    const player = (link ? link.textContent : (whoEl && whoEl.textContent) || '').trim();
    const value = (rowEl.querySelector('.value').textContent || '').trim();
    return [escapeCSVValue(rank), escapeCSVValue(player), escapeCSVValue(value)].join(',');
  });

  return [header, ...rows].join('\n');
}

function escapeCSVValue(value) {
  // If the value contains comma, quote, or newline, wrap it in quotes
  if (value.includes(',') || value.includes('"') || value.includes('\n')) {
    // Escape quotes by doubling them
    value = value.replace(/"/g, '""');
    return `"${value}"`;
  }
  return value;
}

function showCSVDialog(csv) {
  // Create overlay
  const overlay = document.createElement('div');
  overlay.id = 'table-to-csv-overlay';
  
  // Create dialog
  const dialog = document.createElement('div');
  dialog.id = 'table-to-csv-dialog';
  
  dialog.innerHTML = `
    <h3>Table converted to CSV</h3>
    <textarea readonly></textarea>
    <div class="button-group">
      <button class="cancel" id="csv-close">Close</button>
      <button class="secondary" id="csv-copy">Copy to Clipboard</button>
      <button class="primary" id="csv-download">Download CSV</button>
    </div>
  `;
  
  document.body.appendChild(overlay);
  document.body.appendChild(dialog);
  
  const textarea = dialog.querySelector('textarea');
  textarea.value = csv;
  textarea.select();
  
  document.getElementById('csv-close').addEventListener('click', closeDialog);
  document.getElementById('csv-copy').addEventListener('click', () => copyToClipboard(csv));
  document.getElementById('csv-download').addEventListener('click', () => downloadCSV(csv));
  overlay.addEventListener('click', closeDialog);
}

function closeDialog() {
  const dialog = document.getElementById('table-to-csv-dialog');
  const overlay = document.getElementById('table-to-csv-overlay');
  if (dialog) dialog.remove();
  if (overlay) overlay.remove();
}

function copyToClipboard(text) {
  navigator.clipboard.writeText(text).then(() => {
    showStatus('CSV copied to clipboard!');
    closeDialog();
  }).catch(err => {
    showStatus('Failed to copy to clipboard', true);
    console.error('Could not copy text: ', err);
  });
}

function downloadCSV(csv) {
  const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
  const url = URL.createObjectURL(blob);
  const link = document.createElement('a');
  
  // Generate filename with timestamp
  const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
  link.download = `table_${timestamp}.csv`;
  link.href = url;
  
  document.body.appendChild(link);
  link.click();
  document.body.removeChild(link);
  URL.revokeObjectURL(url);
  
  showStatus('CSV downloaded!');
  closeDialog();
}

function showStatus(message, isError = false) {
  // Remove any existing status
  const existing = document.getElementById('table-to-csv-status');
  if (existing) existing.remove();
  
  const status = document.createElement('div');
  status.id = 'table-to-csv-status';
  status.textContent = message;
  if (isError) {
    status.style.backgroundColor = '#f44336';
  }
  
  document.body.appendChild(status);
  
  setTimeout(() => {
    status.remove();
  }, 3000);
}